
#include "Effects.h"
#include "TrigTables.h"
#include <math.h>
#include <string.h>

EffectState g_effects[3];

void Effects_Echo_Init(EffectState* pState, unsigned long sample_rate, unsigned short delay, unsigned short amplitude) {
    pState->echo.transition = 0;
    Effects_Echo_Adjust(pState, sample_rate, delay, amplitude);
    pState->echo.delay = ExpTable[delay>>6];
    pState->echo.amplitude = pState->echo.target_amplitude;
    pState->echo.transition = 0;
}
void Effects_Echo_Adjust(EffectState* pState, unsigned long sample_rate, unsigned short delay, unsigned short amplitude) {
    pState->echo.target_amplitude = amplitude*7/8;
    if( /*(delay > pState->echo.delay+100 || delay < pState->echo.delay-100) &&*/ !pState->echo.transition ) {
        pState->echo.newdelay = ExpTable[delay>>6];
        pState->echo.transition = 1023;
    } else if( pState->echo.transition ) {
        if( --pState->echo.transition == 0 )
            pState->echo.delay = pState->echo.newdelay;
    }
}
signed short Effects_Echo_Apply(EffectState* pState, signed short* Buffer, unsigned long BufferPos, unsigned long BufferSize) {
    int val, old_val, old_pos;
    unsigned long delay;
    delay = BufferSize;
    delay = (delay * pState->echo.delay)>>16;
    old_pos = (int)BufferPos - (int)delay;
    if( old_pos < 0 )
        old_pos += BufferSize;
    val = Buffer[BufferPos];
    old_val = Buffer[old_pos];
    if( pState->echo.transition ) {
        int new_val;
        delay = BufferSize;
        delay = (delay * pState->echo.newdelay)>>16;
        old_pos = (int)BufferPos - (int)delay;
        if( old_pos < 0 )
            old_pos += BufferSize;
        new_val = Buffer[BufferPos];
        val = ((val * pState->echo.transition) + (new_val * (1024 - pState->echo.transition))) >> 10;
    }

    val = (val * (65536 - pState->echo.amplitude) + old_val * pState->echo.amplitude)>>16;
    Buffer[BufferPos] = val;
    if( pState->echo.transition )
        if( --pState->echo.transition == 0 )
            pState->echo.delay = pState->echo.newdelay;
    if( pState->echo.target_amplitude > pState->echo.amplitude )
        ++pState->echo.amplitude;
    else if( pState->echo.target_amplitude < pState->echo.amplitude )
        --pState->echo.amplitude;
    return val;
}
unsigned int Effects_Echo_GetLevel(EffectState* pState) {
    return 65536 - pState->echo.amplitude;
}


void Effects_Reverb_Init(EffectState* pState, unsigned long sample_rate, unsigned short delay, unsigned short amplitude) {
    pState->reverb.transition = 0;
    Effects_Reverb_Adjust(pState, sample_rate, delay, amplitude);
    pState->reverb.delay = ExpTable[delay>>6];
    pState->reverb.amplitude = pState->reverb.target_amplitude;
    pState->reverb.reverb_amplitude = pState->reverb.target_reverb_amplitude;
    pState->reverb.transition = 0;
}
void Effects_Reverb_Adjust(EffectState* pState, unsigned long sample_rate, unsigned short delay, unsigned short amplitude) {
    pState->reverb.target_amplitude = amplitude>>1;
    if( delay != pState->reverb.delay && !pState->reverb.transition ) {
        pState->reverb.newdelay = ExpTable[delay>>6];
        pState->reverb.transition = 1023;
    }
    pState->reverb.target_reverb_amplitude = amplitude;
    if( pState->reverb.target_reverb_amplitude > 32768 )
        pState->reverb.target_reverb_amplitude = 32768;
}
signed short Effects_Reverb_Apply(EffectState* pState, signed short* Buffer, unsigned long BufferPos, unsigned long BufferSize) {
    int val, old_val, old_pos;
    unsigned long delay;

    old_pos = (int)BufferPos - 12;
    if( old_pos < 0 )
        old_pos += BufferSize;
    val = Buffer[BufferPos];
    old_val = Buffer[old_pos];
    val = (val * (65536 - pState->reverb.reverb_amplitude) + old_val * pState->reverb.reverb_amplitude)>>16;

    delay = BufferSize;
    if( pState->reverb.transition ) {
        unsigned long temp = (pState->reverb.delay * pState->reverb.transition + pState->reverb.newdelay * (1024 - pState->reverb.transition)) >> 10;
        delay = (delay * temp)>>16;
    } else {
        delay = (delay * pState->reverb.delay)>>16;
    }
    old_pos = (int)BufferPos - (int)delay;
    if( old_pos < 0 )
        old_pos += BufferSize;
    old_val = Buffer[old_pos];
/*
    if( pState->reverb.transition ) {
        int new_val;

        delay = BufferSize;
        delay = (delay * pState->reverb.newdelay)>>16;
        old_pos = (int)BufferPos - (int)delay;
        if( old_pos < 0 )
            old_pos += BufferSize;
        new_val = Buffer[BufferPos];
        val = ((val * pState->reverb.transition) + (new_val * (1024 - pState->reverb.transition))) >> 10;
    }
*/

    val = (val * (65536 - pState->reverb.amplitude) + old_val * pState->reverb.amplitude)>>16;
    Buffer[BufferPos] = val;
    if( pState->reverb.transition )
        if( --pState->reverb.transition == 0 )
            pState->reverb.delay = pState->reverb.newdelay;

    if( pState->reverb.target_amplitude > pState->reverb.amplitude )
        ++pState->reverb.amplitude;
    else if( pState->reverb.target_amplitude < pState->reverb.amplitude )
        --pState->reverb.amplitude;
    else if( pState->reverb.target_reverb_amplitude > pState->reverb.reverb_amplitude )
        ++pState->reverb.reverb_amplitude;
    else if( pState->reverb.target_reverb_amplitude < pState->reverb.reverb_amplitude )
        --pState->reverb.reverb_amplitude;
    return val;
}
unsigned int Effects_Reverb_GetLevel(EffectState* pState) {
    if( pState->reverb.amplitude == 0 && pState->reverb.reverb_amplitude == 0 )
        return 0x10000;
    else
        return ((unsigned long)(65536 - pState->reverb.amplitude) * (unsigned long)(65536 - pState->reverb.reverb_amplitude)) >> 16;
}


void Effects_Tremolo_Init(EffectState* pState, unsigned long sample_rate, unsigned short depth, unsigned short frequency) {
    /*
     * frequency: 0-65535 => 1-50Hz
     * depth: 0-65535 => 0% to 100% modulation
     */

    pState->tremolo.pos = 0;
    Effects_Tremolo_Adjust(pState, sample_rate, depth, frequency);
    pState->tremolo.sinemult = pState->tremolo.target_sinemult;
}
void Effects_Tremolo_Adjust(EffectState* pState, unsigned long sample_rate, unsigned short depth, unsigned short frequency) {
    pState->tremolo.sineadd = SINESIZE * (frequency * 49 + 65536) / sample_rate;
    pState->tremolo.target_sinemult = depth;
}
signed short Effects_Tremolo_Apply(EffectState* pState, signed short data) {
    unsigned long sineval = (unsigned long)SineTable[(pState->tremolo.pos >> 16)&(SINESIZE-1)]+32768;
    sineval = ((sineval * (unsigned long)pState->tremolo.sinemult)>>16) + (65536 - (unsigned long)pState->tremolo.sinemult);
    data = ((int)data * (int)sineval) >> 16;
    pState->tremolo.pos += pState->tremolo.sineadd;
    if( pState->tremolo.target_sinemult > pState->tremolo.sinemult )
        ++pState->tremolo.sinemult;
    else if( pState->tremolo.target_sinemult < pState->tremolo.sinemult )
        --pState->tremolo.sinemult;
    return data;
}
unsigned int Effects_Tremolo_GetLevel(EffectState* pState) {
    return 65536 - (pState->tremolo.sinemult>>1);
}


void Effects_Vibrato_Init(EffectState* pState, unsigned long sample_rate, unsigned short depth, unsigned short frequency) {
    /*
     * frequency shift: 0-65535 => 1-50Hz
     * depth: 0-65535 => 0 to 2 semitones
     */

    pState->vibrato.pos = 0;
    pState->vibrato.frac = 65536;
    Effects_Vibrato_Adjust(pState, sample_rate, depth, frequency);
    pState->vibrato.sineadd = pState->vibrato.targetadd;
    pState->vibrato.sinemult = pState->vibrato.targetmult;
}
void Effects_Vibrato_Adjust(EffectState* pState, unsigned long sample_rate, unsigned short depth, unsigned short frequency) {
    unsigned long freq, adj_depth;

    /*
     * frequency shift: 0-65535 => 1-50Hz
     * depth: 0-65535 => 0 to 2 semitones
     */

    freq = SINESIZE * 16;
    freq *= frequency;
    pState->vibrato.targetadd = freq / sample_rate;
    adj_depth = depth;
    adj_depth *= 65536 - ((frequency*3)>>2);
    adj_depth >>= 16;
    pState->vibrato.targetmult = (sample_rate / 256 * adj_depth) >> 16;//(8026 * depth) >> 16;
}
signed short Effects_Vibrato_Apply(EffectState* pState, signed short* Buffer, unsigned long BufferPos, unsigned long BufferSize) {
    int offset1, offset2, frac, diff, sinepos;
    unsigned long sineval;
    sinepos = (pState->vibrato.pos >> 16)&(LOGADJSINESIZE-1);
    sineval = (int)LogAdjSineTable[sinepos] + (int)32768;
    if( sinepos == LOGADJSINESIZE-1 ) {
        frac = pState->vibrato.pos&0xFFFF;
        diff = (int)LogAdjSineTable[0] - (int)LogAdjSineTable[sinepos];
        sineval += (diff * frac)>>16;
    } else {
        frac = pState->vibrato.pos&0xFFFF;
        diff = (int)LogAdjSineTable[sinepos+1] - (int)LogAdjSineTable[sinepos];
        sineval += (diff * frac)>>16;
    }
    sineval = 65535 - sineval;
    sineval *= pState->vibrato.sinemult;
    pState->vibrato.pos += pState->vibrato.sineadd;
    offset1 = (int)BufferPos - (sineval>>16);
    offset2 = offset1 - 1;
    frac = sineval&0xFFFF;
    if( offset1 < 0 )
        offset1 += BufferSize;
    if( offset2 < 0 )
        offset2 += BufferSize;
    diff = (int)Buffer[offset2] - (int)Buffer[offset1];
    diff = (diff * frac)>>16;
    if( pState->vibrato.targetadd > pState->vibrato.sineadd )
        ++pState->vibrato.sineadd;
    else if( pState->vibrato.targetadd < pState->vibrato.sineadd )
        --pState->vibrato.sineadd;
    if( pState->vibrato.targetmult > pState->vibrato.sinemult )
        ++pState->vibrato.sinemult;
    else if( pState->vibrato.targetmult < pState->vibrato.sinemult )
        --pState->vibrato.sinemult;
    return Buffer[offset1] + diff;
}
unsigned int Effects_Vibrato_GetLevel(EffectState* pState) {
    return 65536;
}

void Effects_Overdrive_Init(EffectState* pState, unsigned long sample_rate, unsigned short gain, unsigned short softness) {
    Effects_Overdrive_Adjust(pState, sample_rate, gain, softness);
    pState->overdrive.gain = pState->overdrive.target_gain;
    pState->overdrive.softness = pState->overdrive.target_softness;
}
void Effects_Overdrive_Adjust(EffectState* pState, unsigned long sample_rate, unsigned short gain, unsigned short softness) {
    /*
     * gain: 0-65535 => 1-16
     * softness: 0-65535 => 0-100% reduction in gain over unity at full amplitude
     */

    pState->overdrive.target_gain = (65536 + gain * 8 + (((unsigned int)softness * (unsigned int)gain)>>17)) >> 4;
    pState->overdrive.target_softness = ((unsigned int)gain * (unsigned int)softness)>>17;
}
signed short Effects_Overdrive_Apply(EffectState* pState, signed short data) {
    int absval, val, gain;
    val = data;
    absval = val;
    if( absval < 0 )
        absval = -absval;
    gain = pState->overdrive.gain - (((int)pState->overdrive.softness * (int)absval)>>15);
    val = ((int)val * (int)gain) >> 12;
    if( val > 32767 )
        val = 32767;
    else if( val < -32768 )
        val = -32768;

    if( pState->overdrive.target_softness > pState->overdrive.softness )
        ++pState->overdrive.softness;
    else if( pState->overdrive.target_softness < pState->overdrive.softness )
        --pState->overdrive.softness;
    if( pState->overdrive.target_gain > pState->overdrive.gain )
        ++pState->overdrive.gain;
    else if( pState->overdrive.target_gain < pState->overdrive.gain )
        --pState->overdrive.gain;

    return val;
}
unsigned int Effects_Overdrive_GetLevel(EffectState* pState) {
    return 65536;
}


void Effects_Fuzz_Init(EffectState* pState, unsigned long sample_rate, unsigned short gain, unsigned short softness) {
    Effects_Fuzz_Adjust(pState, sample_rate, gain, softness);
    pState->fuzz.gain = pState->fuzz.target_gain;
    pState->fuzz.softness = pState->fuzz.target_softness;
}
void Effects_Fuzz_Adjust(EffectState* pState, unsigned long sample_rate, unsigned short gain, unsigned short softness) {
    /*
     * gain: 0-65535 => 1-16
     * softness: 0-65535 => 0-100% reduction in gain over unity at full amplitude
     */

    pState->fuzz.target_gain = (65536 + gain * 8 + (((unsigned int)softness * (unsigned int)gain)>>17)) >> 4;
    pState->fuzz.target_softness = ((unsigned int)gain * (unsigned int)softness)>>16;
}
signed short Effects_Fuzz_Apply(EffectState* pState, signed short data) {
    int absval, val, gain;
    val = data;
    absval = val;
    if( absval < 0 )
        absval = -absval;
    gain = pState->fuzz.gain - (((int)pState->fuzz.softness * (int)absval)>>15);
    val = ((((int)val + 4096) * (int)gain) >> 12) - 4096;
    if( val > 32767 )
        val = 32767;
    else if( val < -32768 )
        val = -32768;

    if( pState->fuzz.target_softness > pState->fuzz.softness )
        ++pState->fuzz.softness;
    else if( pState->fuzz.target_softness < pState->fuzz.softness )
        --pState->fuzz.softness;
    if( pState->fuzz.target_gain > pState->fuzz.gain )
        ++pState->fuzz.gain;
    else if( pState->fuzz.target_gain < pState->fuzz.gain )
        --pState->fuzz.gain;

    return val;
}
unsigned int Effects_Fuzz_GetLevel(EffectState* pState) {
    return 65536;
}

void Effects_Compression_Init(EffectState* pState, unsigned long sample_rate, unsigned short attack, unsigned short decay) {
    pState->compress.level = 16384 * 16;
    pState->compress.delay = 0;
    Effects_Compression_Adjust(pState, sample_rate, attack, decay);
}
void Effects_Compression_Adjust(EffectState* pState, unsigned long sample_rate, unsigned short attack, unsigned short decay) {
    /*
     * attack: 0-65535 => 10s to 100ms
     * decay: 0-65535 => 100s to 1s
     */

    float attack_factor = log10(1.0 + attack * 9.0 / 65535.0);
    float attack_seconds = 99 * attack_factor + 1;
    float decay_factor = log10(1.0 + decay * 9.0 / 65535.0);
    float decay_seconds = 9.99 * decay_factor + 0.01;
    pState->compress.attack_rate = 16384 * 256 / attack_seconds / sample_rate;
    if( pState->compress.attack_rate < 1 )
        pState->compress.attack_rate = 1;
    pState->compress.decay_rate = 16384 * 256 / decay_seconds / sample_rate;
    if( pState->compress.decay_rate < 1 )
        pState->compress.decay_rate = 1;
    pState->compress.maxdelay = sample_rate / 100;
}
signed short Effects_Compression_Apply(EffectState* pState, signed short data) {
    int val;

    val = (data * (pState->compress.level>>8)) >> 14;
    if( val > -24576 && val < 24576 ) {
        if( pState->compress.delay >= pState->compress.maxdelay ) {
            pState->compress.level += pState->compress.attack_rate;
            if( pState->compress.level > 262143*256 )
                pState->compress.level = 262143*256;
        } else {
            ++pState->compress.delay;
        }
    } else {
        if( val > 32767 - 8192 || val < -32768 + 8192 ) {
            if( val > 32767 ) {
                val = 32767;
                pState->compress.level -= pState->compress.decay_rate<<6;
            } else if( val < -32768 ) {
                val = -32768;
                pState->compress.level -= pState->compress.decay_rate<<6;
            } else {
                pState->compress.level -= pState->compress.decay_rate;
            }
            if( pState->compress.level < 16384*256 )
                pState->compress.level = 16384*256;
        }
        pState->compress.delay = 0;
    }
    return val;
}
unsigned int Effects_Compression_GetLevel(EffectState* pState) {
    return 65536 - 1024*2;
}

void Effects_NoiseGate_Init(EffectState* pState, unsigned long sample_rate, unsigned short threshold, unsigned short hysteresis) {
    Effects_NoiseGate_Adjust(pState, sample_rate, threshold, hysteresis);
    pState->noisegate.delay = 0;
    pState->noisegate.gain = 1024;
}
void Effects_NoiseGate_Adjust(EffectState* pState, unsigned long sample_rate, unsigned short threshold, unsigned short hysteresis) {
    /*
     * threshold: 0-65535 => 0-50%
     * hysteresis: 0-65535 => 0-75%
     */

    pState->noisegate.max_threshold = threshold >> 1;
    pState->noisegate.min_threshold = (pState->noisegate.max_threshold * (65536 - hysteresis * 3 / 4)) >> 16;
    pState->noisegate.maxdelay = sample_rate / 100;
}
signed short Effects_NoiseGate_Apply(EffectState* pState, signed short data) {
    int val, threshold;

    threshold = pState->noisegate.delay >= pState->noisegate.maxdelay ? pState->noisegate.max_threshold : pState->noisegate.min_threshold;
    if( data > threshold || data < -threshold ) {
        pState->noisegate.delay = 0;
        if( pState->noisegate.gain < 1024 )
            ++pState->noisegate.gain;
    } else {
        if( pState->noisegate.delay >= pState->noisegate.maxdelay ) {
            if( pState->noisegate.gain > 0 )
                --pState->noisegate.gain;
        } else {
            ++pState->noisegate.delay;
        }
    }

    val = data;
    val = (val * pState->noisegate.gain) >> 10;
    return val;
}
unsigned int Effects_NoiseGate_GetLevel(EffectState* pState) {
    return 65536;
}


void Effects_Flange_Init(EffectState* pState, unsigned long sample_rate, unsigned short depth, unsigned short frequency) {
    /*
     * frequency shift: 0-65535 => 1-50Hz
     * depth: 0-65535 => 0 to 2 semitones
     */

    pState->flange.pos = 0;
    pState->flange.frac = 65536;
    Effects_Flange_Adjust(pState, sample_rate, depth, frequency);
    pState->flange.sineadd = pState->flange.targetadd;
    pState->flange.sinemult = pState->flange.targetmult;
    pState->flange.flangedepth = 32768;//depth;
}
void Effects_Flange_Adjust(EffectState* pState, unsigned long sample_rate, unsigned short depth, unsigned short frequency) {
    unsigned long freq, adj_depth;

    /*
     * frequency shift: 0-65535 => 1-50Hz
     * depth: 0-65535 => 0 to 2 semitones
     */

    freq = SINESIZE * 4;
    freq *= frequency;
    pState->flange.targetadd = freq / sample_rate;
    adj_depth = depth;
    adj_depth *= 65536 - ((frequency*3)>>2);
    adj_depth >>= 16;
    pState->flange.targetmult = (sample_rate / 256 * adj_depth) >> 16;//(8026 * depth) >> 16;
    pState->flange.flangedepth = 32768;//depth;
}
signed short Effects_Flange_Apply(EffectState* pState, signed short* Buffer, unsigned long BufferPos, unsigned long BufferSize) {
    int offset1, offset2, frac, diff, sinepos, val;
    unsigned long sineval;
    sinepos = (pState->flange.pos >> 16)&(LOGADJSINESIZE-1);
    sineval = (int)LogAdjSineTable[sinepos] + (int)32768;
    if( sinepos == LOGADJSINESIZE-1 ) {
        frac = pState->flange.pos&0xFFFF;
        diff = (int)LogAdjSineTable[0] - (int)LogAdjSineTable[sinepos];
        sineval += (diff * frac)>>16;
    } else {
        frac = pState->flange.pos&0xFFFF;
        diff = (int)LogAdjSineTable[sinepos+1] - (int)LogAdjSineTable[sinepos];
        sineval += (diff * frac)>>16;
    }
    sineval = 65535 - sineval;
    sineval *= pState->flange.sinemult;
    pState->flange.pos += pState->flange.sineadd;
    offset1 = (int)BufferPos - (sineval>>16);
    offset2 = offset1 - 1;
    frac = sineval&0xFFFF;
    if( offset1 < 0 )
        offset1 += BufferSize;
    if( offset2 < 0 )
        offset2 += BufferSize;
    diff = (int)Buffer[offset2] - (int)Buffer[offset1];
    diff = (diff * frac)>>16;
    if( pState->flange.targetadd > pState->flange.sineadd )
        ++pState->flange.sineadd;
    else if( pState->flange.targetadd < pState->flange.sineadd )
        --pState->flange.sineadd;
    if( pState->flange.targetmult > pState->flange.sinemult )
        ++pState->flange.sinemult;
    else if( pState->flange.targetmult < pState->flange.sinemult )
        --pState->flange.sinemult;
    val = Buffer[offset1] + diff;
    diff = val - (int)Buffer[BufferPos];
    return Buffer[BufferPos] + ((diff * pState->flange.flangedepth)>>16);
}
unsigned int Effects_Flange_GetLevel(EffectState* pState) {
    return 65536;
}


void Effects_Compute2ndOrderAllpassCoefficients_FixedPoint(unsigned long freq, unsigned long sample_rate, unsigned short Q, IIR_Coeffs* pCoeffs) {
    int _pos, _frac, _g, _d, _K, _b0, _b1;
    _pos = freq * 1024 / sample_rate;
    _frac = freq * 1024 % sample_rate;
    _g = (TanTable[_pos] * (sample_rate - _frac) + TanTable[_pos+1] * (_frac)) / sample_rate;
    _d = 0x800000/Q;
    _K = 0x40000000 / (0x8000 + (((_d+_g)*_g)>>15));
    _b0 = ((0x8000 - ((_g*(_d - _g))>>15)) * _K)>>15;
    _b1 = (2 * (((_g*_g)>>15) - 0x8000) * _K)>>15;
    pCoeffs->a1 = _b1;
    pCoeffs->a2 = _b0;
}
signed short Effects_Process2ndOrderAllpass(signed short value, const IIR_Coeffs* pCoeffs, IIR_State* pState) {
    int temp = (value - ((pCoeffs->a1 * pState->state1)>>15) - ((pCoeffs->a2 * pState->state2)>>15))>>1;
    int ret = ((pCoeffs->a2 * temp)>>15) + ((pCoeffs->a1 * pState->state1)>>15) + (pState->state2<<1);
    pState->state2 = pState->state1;
    if( temp > 32767 )
        temp = 32767;
    else if( temp < -32768 )
        temp = -32768;
    pState->state1 = temp;
    if( ret > 32767 )
        ret = 32767;
    else if( ret < -32768 )
        ret = -32768;
    return ret;
}
void Effects_Phaser_Init(EffectState* pState, unsigned long sample_rate, unsigned short depth, unsigned short frequency) {
//    Effects_Compute2ndOrderAllpassCoefficients(5000, sample_rate, 181, &pState->phaser.coeffs);
//    Effects_Compute2ndOrderAllpassCoefficients_FixedPoint(freq, sample_rate, 181, &pState->phaser.coeffs);
//    memset(&pState->phaser.states, 0, sizeof(pState->phaser.states));
    Effects_Phaser_Adjust(pState, sample_rate, depth, frequency);
    pState->phaser.pos = 0;
    pState->phaser.sample_rate = sample_rate;
}
void Effects_Phaser_Adjust(EffectState* pState, unsigned long sample_rate, unsigned short depth, unsigned short frequency) {
    unsigned long freq = frequency;
    freq *= sample_rate;
    freq >>= 18;
//    Effects_Compute2ndOrderAllpassCoefficients_FixedPoint(freq, sample_rate, 256/*181*/, &pState->phaser.coeffs);
    pState->phaser.mix = depth/2;

    freq = SINESIZE * 16;
    freq *= frequency;
    pState->phaser.sineadd = freq / sample_rate;
}
IIR_State iir_states[4];
signed short Effects_Phaser_Apply(EffectState* pState, signed short data) {
    int val, frac, diff, sinepos;
    unsigned long sineval;//, freq;
    IIR_Coeffs coeffs;

    sinepos = (pState->phaser.pos >> 16)&(LOGADJSINESIZE-1);
    sineval = (int)LogAdjSineTable[sinepos] + (int)32768;
    if( sinepos == LOGADJSINESIZE-1 ) {
        frac = pState->phaser.pos&0xFFFF;
        diff = (int)LogAdjSineTable[0] - (int)LogAdjSineTable[sinepos];
        sineval += (diff * frac)>>16;
    } else {
        frac = pState->phaser.pos&0xFFFF;
        diff = (int)LogAdjSineTable[sinepos+1] - (int)LogAdjSineTable[sinepos];
        sineval += (diff * frac)>>16;
    }
    pState->phaser.pos += pState->phaser.sineadd;
    sineval *= pState->phaser.sample_rate;
    sineval >>= 18;

    Effects_Compute2ndOrderAllpassCoefficients_FixedPoint(sineval, pState->phaser.sample_rate, 256/*181*/, &coeffs);
    val = Effects_Process2ndOrderAllpass(data, &coeffs, &iir_states[0]);
    val = Effects_Process2ndOrderAllpass(val, &coeffs, &iir_states[1]);
    val = Effects_Process2ndOrderAllpass(val, &coeffs, &iir_states[2]);
    val = Effects_Process2ndOrderAllpass(val, &coeffs, &iir_states[3]);
    return data + (((val - data) * pState->phaser.mix)>>16);
}
unsigned int Effects_Phaser_GetLevel(EffectState* pState) {
    return 65536;
}


void Effects_Init(EffectState* pState, unsigned long sample_rate, unsigned short param1, unsigned short param2) {
    switch(pState->cur_effect) {
        case Echo:
            Effects_Echo_Init(pState, sample_rate, param1, param2);
            break;
        case Reverb:
            Effects_Reverb_Init(pState, sample_rate, param1, param2);
            break;
        case Tremolo:
            Effects_Tremolo_Init(pState, sample_rate, param1, param2);
            break;
        case Vibrato:
            Effects_Vibrato_Init(pState, sample_rate, param1, param2);
            break;
        case Overdrive:
            Effects_Overdrive_Init(pState, sample_rate, param1, param2);
            break;
        case Fuzz:
            Effects_Fuzz_Init(pState, sample_rate, param1, param2);
            break;
        case Compression:
            Effects_Compression_Init(pState, sample_rate, param1, param2);
            break;
        case NoiseGate:
            Effects_NoiseGate_Init(pState, sample_rate, param1, param2);
            break;
        case Flange:
            Effects_Flange_Init(pState, sample_rate, param1, param2);
            break;
        case Phaser:
            Effects_Phaser_Init(pState, sample_rate, param1, param2);
            break;
    }
}
void Effects_Adjust(EffectState* pState, unsigned long sample_rate, unsigned short param1, unsigned short param2) {
    switch(pState->cur_effect) {
        case Echo:
            Effects_Echo_Adjust(pState, sample_rate, param1, param2);
            break;
        case Reverb:
            Effects_Reverb_Adjust(pState, sample_rate, param1, param2);
            break;
        case Tremolo:
            Effects_Tremolo_Adjust(pState, sample_rate, param1, param2);
            break;
        case Vibrato:
            Effects_Vibrato_Adjust(pState, sample_rate, param1, param2);
            break;
        case Overdrive:
            Effects_Overdrive_Adjust(pState, sample_rate, param1, param2);
            break;
        case Fuzz:
            Effects_Fuzz_Adjust(pState, sample_rate, param1, param2);
            break;
        case Compression:
            Effects_Compression_Adjust(pState, sample_rate, param1, param2);
            break;
        case NoiseGate:
            Effects_NoiseGate_Adjust(pState, sample_rate, param1, param2);
            break;
        case Flange:
            Effects_Flange_Adjust(pState, sample_rate, param1, param2);
            break;
        case Phaser:
            Effects_Phaser_Adjust(pState, sample_rate, param1, param2);
            break;
    }
}
signed short Effects_Apply(EffectState* pState, signed short* Buffer, unsigned long BufferPos, unsigned long BufferSize) {
    switch(pState->cur_effect) {
        case Echo:
            return Effects_Echo_Apply(pState, Buffer, BufferPos, BufferSize);
        case Reverb:
            return Effects_Reverb_Apply(pState, Buffer, BufferPos, BufferSize);
        case Tremolo:
            return Effects_Tremolo_Apply(pState, Buffer[BufferPos]);
        case Vibrato:
            return Effects_Vibrato_Apply(pState, Buffer, BufferPos, BufferSize);
        case Overdrive:
            return Effects_Overdrive_Apply(pState, Buffer[BufferPos]);
        case Fuzz:
            return Effects_Fuzz_Apply(pState, Buffer[BufferPos]);
        case Compression:
            return Effects_Compression_Apply(pState, Buffer[BufferPos]);
        case NoiseGate:
            return Effects_NoiseGate_Apply(pState, Buffer[BufferPos]);
        case Flange:
            return Effects_Flange_Apply(pState, Buffer, BufferPos, BufferSize);
        case Phaser:
            return Effects_Phaser_Apply(pState, Buffer[BufferPos]);
    }
    return 0;
}
unsigned int Effects_GetLevel(EffectState* pState) {
    switch(pState->cur_effect) {
        case Echo:
            return Effects_Echo_GetLevel(pState);
        case Reverb:
            return Effects_Reverb_GetLevel(pState);
        case Tremolo:
            return Effects_Tremolo_GetLevel(pState);
        case Vibrato:
            return Effects_Vibrato_GetLevel(pState);
        case Overdrive:
            return Effects_Overdrive_GetLevel(pState);
        case Fuzz:
            return Effects_Fuzz_GetLevel(pState);
        case Compression:
            return Effects_Compression_GetLevel(pState);
        case NoiseGate:
            return Effects_NoiseGate_GetLevel(pState);
        case Flange:
            return Effects_Flange_GetLevel(pState);
        case Phaser:
            return Effects_Phaser_GetLevel(pState);
    }
    return 0x10000;
}
